//----------------------------------------------------------------------------
//
// Copyright (C) Sartorius Stedim Data Analytics AB 2017 -
//
// Use, modification and distribution are subject to the Boost Software
// License, Version 1.0. (See http://www.boost.org/LICENSE_1_0.txt)
//
//----------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Text;
using SIMCAQLib;

namespace SQPCSharpSample
{
    class EzQ
    {
        public EzQ()
        {
            try
            {
                simcaq = new SIMCAQ();                
                project = null;
                model = null;
                components = null;
                totalComponents = null;
                numComponents = -1;
                numTotalComponents = -1;
                dModXCrit = -1;
                hotellingsT2Crit = -1;
                probLevel = -1;
            }
            catch (System.Exception /*e*/)
            {
                System.Windows.Forms.MessageBox.Show("SIMCA-Q is not registered as a COM DLL, or architecture mismatch"); // Can be that x64 application tries to use an x86 DLL.
            }
            
        }

        // Open the project.
        /// @param projectPath    In: is the full path to the file.
        /// @param projectPassword   In: The password of the file. Can be NULL if not used.
        /// @return             Returns true if successful
        public bool OpenProject(string projectPath, string projectPassword)
        {
            if (project != null)
            {
                project.DisposeProject();
                project = null;
            }
            try
            {
                project = (Project)simcaq.OpenProject(projectPath, projectPassword);

                // TODO Uncomment the line below to allow python plugins.
                // But only if you have complete control of the ones uploading the projecs and know they can be trusted.
                // project.EnableEmbeddedScripts();
            }
            catch (System.Exception e)
            {
                if (e is System.Runtime.InteropServices.COMException)
                    lastErrorMessage = simcaq.GetErrorDescription(((System.Runtime.InteropServices.COMException)e).ErrorCode);
                return false;
            }
            
            return true;
        }

        /// Set the index of the model to use. By default the first model is used.
        /// @param modelIndex  In: The index of the model starting from 1 
        /// @return             Returns true if successful
        public bool SelectModel(int modelIndex)
        {
            try
            {
                System.Diagnostics.Debug.Assert(project != null);

                //project.ChangeMode(DllMode.eModePrediction);

                //////////////////////////////////////////////////////////////////////////
                // Check that the model exists in SIMCAQ
                int nModels = project.GetNumberOfModels();

                // Make sure that vi have some models
                if (nModels < 1)
                    return false;

                // Get the model number connected with the input index.
                int modelNumber = project.GetModelNumberFromIndex(modelIndex);
                model = (Model)project.GetModel(modelNumber);

                // Check if model is correct (=fitted) 
                bool isFitted = model.IsModelFitted() != 0;
                if (!isFitted)
                    return false;

                // Get results
                numComponents = model.GetNumberOfComponents();
                numTotalComponents = model.GetNumberOfPredictiveComponents() + model.GetNumberOfXOrthogonalComponents();

                // Set components to use
                components = (IntVector)simcaq.GetNewIntVector(1);
                components.SetData(1, numComponents);

                totalComponents = (IntVector)simcaq.GetNewIntVector(1);
                totalComponents.SetData(1, numTotalComponents);
                
                probLevel = model.GetDefaultProbabilityLevel();
                dModXCrit = model.GetDModXCrit(numComponents, NormalizedState.eNormalized_True, probLevel);
                hotellingsT2Crit = model.GetT2RangeCrit(1, numTotalComponents, probLevel);
            }
            catch (System.Exception e)
            {
                if (e is System.Runtime.InteropServices.COMException)
                    lastErrorMessage = simcaq.GetErrorDescription(((System.Runtime.InteropServices.COMException)e).ErrorCode);
                return false;
            }

            return true;
        }

        /// Set the data of the prediction set to use. checks variable names so we now that the correct prediction is made
        /// @param quantitativeMatrix  In: A matrix with quantitative prediction data, row by row
        /// @param quantRowSize            In: Number of rows in quantitative prediction data
        /// @param quantColSize            In: Number of columns in the quantitative prediction data
        /// @param columnNames             In: vector with the names of the columns
        /// @return             Returns true if successful
        public bool Predict(float[] quantitativeMatrix, int quantRowSize, int quantColSize, string[] columnNames)
        {
            try
            {
                // Declare variables
                PreparePrediction prepPred = (PreparePrediction)model.PreparePrediction(); // Get a prepare prediction object to set data for the prediction
                VariableVector varVec = (VariableVector)prepPred.GetVariablesForPrediction();
                int numPredsetVars = varVec.GetSize();

                // TODO, cash  this if you can and know that the data is always in the same order, alternatively create a lookup of "varVec".
                Dictionary<string, int> DataLookup = new Dictionary<string, int>();
                for (int iCol = 0; iCol < quantColSize; ++iCol)
                    DataLookup[columnNames[iCol]] = iCol;


                for (int iCol = 1; iCol <= numPredsetVars; ++iCol)
                {
                    Variable var = (Variable)varVec.GetVariable(iCol);
                    bool bIsQual = var.IsQualitative() != 0;
                    String strName = var.GetName(1);
                    if (DataLookup.ContainsKey(strName))
                    {
                        int iDataCol = DataLookup[strName];
                        for (int iRow = 0; iRow < quantRowSize; ++iRow)
                        {
                            if (bIsQual)
                            {
                                // not implemented in this sample
                                return false;
                                // prepPred.SetQualitativeData(iRow+1, iCol, qualitativeMatrix[iRow * qualColSize + iDataCol]);
                            }
                            else
                            {
                                prepPred.SetQuantitativeData(iRow + 1, iCol, quantitativeMatrix[iRow * quantColSize + iDataCol]);
                            }
                        }
                    }
                    else
                    {
                        // missing a variable
                        // predictions can be calculated anyway if at least a few of the variables exists,
                        // change here to return true if you like to predict even if you don't have all the data.
                        return false;
                    }
                }

                // Predict
                prediction = (Prediction)prepPred.GetPrediction();
            }
            catch (System.Exception e)
            {
                if (e is System.Runtime.InteropServices.COMException)
                    lastErrorMessage = simcaq.GetErrorDescription(((System.Runtime.InteropServices.COMException)e).ErrorCode);
                return false;
            }

            return true;
        }

        /// Set the data of the prediction set to use. Assumes all data is in the correct order already, no name checking.
        /// @param quantitativeMatrix  In: A matrix with quantitative prediction data, row by row
        /// @param quantRowSize            In: Number of rows in quantitative prediction data
        /// @param quantColSize            In: Number of columns in the quantitative prediction data
        /// @param qualitativeMatrix  In: A matrix with qualitative prediction data, row by row
        /// @param qualRowSize             In: Number of rows in qualitative prediction data
        /// @param qualColSize            In: Number of columns in the qualitative prediction data
        /// @return             Returns true if successful
        public bool Predict2(float[] quantitativeMatrix, int quantRowSize, int quantColSize, string[] qualitativeMatrix, int qualRowSize, int qualColSize)
        {
            try
            {
                // Declare variables
                PreparePrediction prepPred = (PreparePrediction)model.PreparePrediction(); // Get a prepare prediction object to set data for the prediction
                VariableVector varVec = (VariableVector)prepPred.GetVariablesForPrediction();
                int numPredsetVars = varVec.GetSize();

                if (quantitativeMatrix != null && qualitativeMatrix != null && quantRowSize != qualRowSize) // Must be same number of rows
                    return false;

                // Set the data for the prediction
                if ((quantitativeMatrix != null && qualitativeMatrix != null)) // Must do some kind of alignment
                {
                        
                    for (int iCol = 1; iCol <= numPredsetVars; ++iCol)
                    {
                        int iQualPos = 0;
                        int iQuantPos = 0;
                        Variable var = (Variable)varVec.GetVariable(iCol);
                        bool bIsQual = var.IsQualitative() != 0;
                        for (int iRow = 0; iRow < quantRowSize; ++iRow)
                        {
                            if (bIsQual)
                            {
                                prepPred.SetQualitativeData(iRow+1, iCol, qualitativeMatrix[iRow * qualColSize + iQualPos]);
                                iQualPos++;
                            }
                            else
                            {
                                prepPred.SetQuantitativeData(iRow+1, iCol, quantitativeMatrix[iRow * quantColSize + iQuantPos]);
                                iQuantPos++;
                            }
                        }
                    }
                }
                else if (quantitativeMatrix != null)
                {
                    FloatMatrix obsRawData = (FloatMatrix)simcaq.GetNewFloatMatrix(quantRowSize, quantColSize);
                    obsRawData.SetMatrix(ref quantitativeMatrix[0]);
                    prepPred.SetQuantitativeDataMatrix(obsRawData);
                }
                else if (qualitativeMatrix != null)
                {
                    StringMatrix obsRawData = (StringMatrix)simcaq.GetNewStringMatrix(qualRowSize, qualColSize);
                    obsRawData.SetMatrix(ref qualitativeMatrix[0]);
                    prepPred.SetQualitativeDataMatrix(obsRawData);
                }
                        
                // Predict
                prediction = (Prediction)prepPred.GetPrediction();
            }
            catch (System.Exception e)
            {
                if (e is System.Runtime.InteropServices.COMException)
                    lastErrorMessage = simcaq.GetErrorDescription(((System.Runtime.InteropServices.COMException)e).ErrorCode);
                return false;
            }

            return true;
        }

        /// Get predicted T (scores)
        /// @return             Returns the result or null if not successful
        public VectorData GetTPS()
        {
            System.Diagnostics.Debug.Assert(project != null);
            System.Diagnostics.Debug.Assert(prediction != null);

            try
            {
                return (VectorData)prediction.GetTPS(components);
            }
            catch (System.Exception e)
            {
                if (e is System.Runtime.InteropServices.COMException)
                    lastErrorMessage = simcaq.GetErrorDescription(((System.Runtime.InteropServices.COMException)e).ErrorCode);
                return null;
            }
        }

        /// Get T (scores)
        /// @return             Returns the result or null if not successful
        public VectorData GetT()
        {
            System.Diagnostics.Debug.Assert(project != null);
            System.Diagnostics.Debug.Assert(model != null);

            try
            {
                return (VectorData)model.GetT(components);
            }
            catch (System.Exception e)
            {
                if (e is System.Runtime.InteropServices.COMException)
                    lastErrorMessage = simcaq.GetErrorDescription(((System.Runtime.InteropServices.COMException)e).ErrorCode);
                return null;
            }
        }

        /// Get predicted DModX
        /// @return             Returns the result or null if not successful
        public VectorData GetDModXPS()
        {
            System.Diagnostics.Debug.Assert(project != null);
            System.Diagnostics.Debug.Assert(prediction != null);
            try
            {
                return (VectorData)prediction.GetDModXPS(components, NormalizedState.eNormalized_True, ModelingPowerWeightedState.eModelingPowerWeighted_False);
            }
            catch (System.Exception e)
            {
                if (e is System.Runtime.InteropServices.COMException)
                    lastErrorMessage = simcaq.GetErrorDescription(((System.Runtime.InteropServices.COMException)e).ErrorCode);
                return null;
            }
        }

        /// Get predicted Y
        /// @return             Returns the result or null if not successful
        public VectorData GetYPredPS()
        {
            System.Diagnostics.Debug.Assert(project != null);
            System.Diagnostics.Debug.Assert(prediction != null);
            try
            {
                return (VectorData)prediction.GetYPredPS(numComponents, UnscaledState.eUnscaled_False, BacktransformedState.eBacktransformed_True, null);
            }
            catch (System.Exception e)
            {
                if (e is System.Runtime.InteropServices.COMException)
                    lastErrorMessage = simcaq.GetErrorDescription(((System.Runtime.InteropServices.COMException)e).ErrorCode);
                return null;
            }
        }

        /// Get predicted T2
        /// @return             Returns the result or null if not successful
        public VectorData GetT2RangePS()
        {
            System.Diagnostics.Debug.Assert(project != null);
            System.Diagnostics.Debug.Assert(prediction != null);
            try
            {
                return (VectorData)prediction.GetT2RangePS(1, numTotalComponents);
            }
            catch (System.Exception e)
            {
                if (e is System.Runtime.InteropServices.COMException)
                    lastErrorMessage = simcaq.GetErrorDescription(((System.Runtime.InteropServices.COMException)e).ErrorCode);
                return null;
            }
        }

        /// Get the model T2Range critical limit
        /// @return             Returns the result or null if not successful
        public float GetT2RangeCrit()
        {
            System.Diagnostics.Debug.Assert(project != null);
            try
            {
                return model.GetT2RangeCrit(1, numTotalComponents, -1);
            }
            catch (System.Exception e)
            {
                if (e is System.Runtime.InteropServices.COMException)
                    lastErrorMessage = simcaq.GetErrorDescription(((System.Runtime.InteropServices.COMException)e).ErrorCode);
                return 0;
            }
        }

        /// Close the project. This is also done automatically by the 
        /// destructor when the object goes out of scope.
        /// @return             Returns non zero if no error.
        public bool CloseProject()
        {
           if (project != null)
           {
              project.DisposeProject();
              project = null;
           }
           return true;
        }

        protected SIMCAQ simcaq;
        protected Project project;

        public Project Project
        {
            get
            {
                return project;
            }
        }
        protected Prediction prediction;
        public Prediction Prediction
        {
            get
            {
                return prediction;
            }
        }
        protected Model model;
        public Model Model
        {
            get
            {
                return model;
            }
        }
        public String LastErrorMessage
        {
            get
            {
                return lastErrorMessage;
            }
        }

        protected IntVector components;
        protected IntVector totalComponents;
        protected int numComponents;
        protected int numTotalComponents;
        protected float dModXCrit;
        protected float hotellingsT2Crit;
        protected float probLevel;
        protected String lastErrorMessage;
    }
}
